/**
 *
 * \file        ir_driver.h
 *
 * \brief       IR Driver class Definitions
 *
 * \author      Dennis Hromin
 *
 * \date        09/12/2008
 *
 */
#ifndef _IRDRIVER_H_
#define _IRDRIVER_H_
#include "nutypedefs.h"

class IrDriver;
extern IrDriver * g_IrDriver;   //global pointer to IrDriver class

// parameters for Transmitter tasks and queues
#define IR_QUEUE_LENGTH             16  //Must be an even number to handle start and stop IR packets without dropping a pair
#define IR_TASK_PRIORITY            0  //No Priorities setup currently, default is 0
#define IR_TASK_STACK               (32/sizeof(UINT32))

/*NOTE: Max Decode buffer is set to worst possible case when decoding Crestron .IR Packets with
  full set of "header" and "repeater" codes which is 256 time words each x 4 is 1024 timewords total */
#define IR_MAX_DECODE_BUFFER 512        //Defined decode buffer array of 16-bit words = 1024 total bytes allocated

#define IR_MAX_NUM_TIMEWORDS 16         //We MUST support 16 timing words as per cresnet IR spec

// The next define is for the max number of serial over IR data bytes which can be sent at one
// time using a single DMA cycle....
// Each serial byte can take upto 15 words at the max
// ( 1 start bit + 8 bits + 1 Parity + 2 Stop bits + 3 pause bits = 15 bit) and each bit is
// represented by a word so we have 15 words per byte. The max length of the IR data buffer
// used by the DMA engine is 2048 bytes so we can fit in 34 bytes at the max.
#define IR_MAX_SERIAL_BYTES_SENT_IN_ONE_CYCLE  40 // (UINT16)(IR_MAX_DECODE_BUFFER/12)
#define IR_FIXED_PACING_SPACE_BITS              100             //set to add 100 extra char bit times to end of a serial byte

 /*-----------------3/23/2005 11:35AM----------------
 *  IR PACKET DEFINITIONS
 * --------------------------------------------------*/
#define IRSTART      0  // old IR style data
#define IRSTOP       1
#define IRSTROBE     2  // old IR style data
#define IRSERIAL     3
#define IRCONTROL    4
#define IRSTART_1    5  // new format IR data
#define IRSTROBE_1   6  // new format IR data

#define CARRIER_4  1120000
#define CARRIER_5  1000000
#define CARRIER_9  455000
#define CARRIER_11 366000
#define CARRIER_12 336000

#define IR_PARITY_NONE 'n'
#define IR_PARITY_EVEN 'e'
#define IR_PARITY_Z    's'
#define IR_PARITY_ODD  'o'

#define IR_TRANSFER_CODE    0
#define IR_TRANSFER_SERIAL  1

#define IR_MODULATE_ENABLED  0
#define IR_MODULATE_DISABLED 1

#ifndef TRUE
#define TRUE  1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef IR_DECODE_SUCCESS
        #define IR_DECODE_SUCCESS       0
#endif
        #define IR_DECODE_FAIL          1

#define IR_SERIAL_DECODE_SUCCESS        0
#define IR_SERIAL_DECODE_BUSY           1
#define IR_SERIAL_DECODE_FAIL           2


#define IR_DRIVER_READY  0
#define IR_DRIVER_BUSY   1

#define IR_CODETYPE_PULSE                       0
#define IR_CODETYPE_REPEATING           1
#define IR_CODETYPE_SERIAL                      2


/*-----------------3/23/2005 11:35AM----------------
 *  DMA STRUCTURES AND TYPES
 * --------------------------------------------------*/
typedef struct sIrPacketStatus  //This structure is holds current ir data
{
    UINT16 headerlength;  //must be 16-bit variable as cresnet based value is doubled to get actual bytes
    UINT16 repeatlength;  //must be 16-bit variable as cresnet based value is doubled to get actual bytes
    UINT16 decoded_irdata_buffer[IR_MAX_DECODE_BUFFER];
    UINT8 numtimewords;
    UINT32 carrier_freq;
    UINT8 resends;
    UINT8 sendforever;
    UINT8 modulate;
    UINT8 busy;
    UINT8 port;
//  UINT8 source;
    UINT8 transfertype;
}
IR_PACKET_STRUCT;

// old style IR data header for packet decoder
typedef struct
{
    UINT8 length;
    UINT8 button;
    UINT8 carrier;
    UINT8 header_length;
    UINT8 repeat_length;
    UINT8 number_of_timewords;
    UINT16 timeword[16];
}
IRPACKETHDR, *pIRPACKETHDR;


// IR Driver Status States & Errors
typedef enum
{
// NOTICES
    IR_SUCCESS=0,
    IR_BUSY,

// ERRORS & WARNINGS
    IR_INVALID_PACKET,   //Driver called with invalid packet data
    IR_DISCARDED_PACKET,

}ERR_IRDRIVER;


// The IR Generate class is a generic driver for generating an IR pulse.  It runs as a task
// which periodically wakes up to monitor the slave buffer feedback interrupts, etc.
class IrDriver
{
public:
    // create object with its configuration and assign it to its stream
    IrDriver();  //The constructor func definition
    ~IrDriver(); //The destructor func definition
    virtual BOOL IrHwStartGenerateOutput(void) {return 0;};
    virtual void IrHwStopGenerateOutput(void) {};
    virtual void IrHwForceStopGenerateOutput(void) {};
    virtual BOOL IrHwGetLowLevelDriverStatus(void) {return 0;};

    void IrProcessTask(void); // Setup the main IR Process task
    UINT8 ParseIrPacket( UINT8* pCnetIrPacket );
    BOOL DecodeIrPacket( UINT8* pCnetIrPacket );  //Main Decoding task after msg is completely parsed
    BOOL DecodeIrSerialPacket( UINT8* pCnetIrPacket );  //Main (1st) Decoding task after msg is completely parsed
    void InitIrDataPacketStruct(void);
    UINT8 GetIRCodeType(void); //Returns the current IR Packet Type Being decoded. This is used in some low level IR drivers to check status.

    UINT32 m_IrQueue;                   //Handle for the IR message queue
    IR_PACKET_STRUCT m_IrPkt; //Pointer to struct that has current Ir Packet Information
    UINT8   m_SerialBytesLeft;

    BOOL m_bIrSerialStart; //Bool designating the start of a segmented serial transfer

private:
    // class variables
    BOOL   m_MsgAvailable;
    UINT32 m_IrTask;                   //Handle for the IR task

    void AddSerialBit(UINT8 bit_value, UINT16 Timeword);
    void ResetSerial(void) {m_TimingWord=0; m_CurrentBit=0;}
    UINT16  m_TimingWord;
    UINT8   m_CurrentBit;
    UINT16* m_OutPtr;
    UINT8*  m_pSerialStrn;
};

typedef enum {
    ERR_IR_TIMEOUT,     // IR processor timed out
    ERR_IR_UNSUPPORTED_TIMER_CFG,   // unsupported TIMER configuration
    ERR_IR_CONTROLLER_TIMEOUT,      // IR controller did not send configuration packet on startup
    ERR_IR_HW_RESET_FAIL,      // reset of IR controller failed
} ERR_IR;


typedef enum {
    CONFIG_DEFAULT, // Default IR controller configuration
    CONFIG_ALT1,    // Alternate configuration with different pin mapping
    CONFIG_ALT2     // Alternate configuration with different pin mapping
} IR_CONTROLLER_CONFIG;

// Function to Initialize the IR Driver Class from main loop
void IrDriver_Init(void);
//Function that sends back a cresnet message on Ir Busy status from driver in crestcore //
extern void CresnetSlaveSendIrBusy( UINT32 join, UINT8 bValue);
extern const UINT8 g_bIRQueueLength;

#endif  // _IRDRIVER_H_
